import string
import numpy as np
import math
import os

#NOTE:Coded in python version 3.4 (at least up to V0pt4, 3.5 onwards)
#History
#Oct 2016: Added the write_xyz class function. Creates a .xyz file with geometry
#Jan 2017:V0pt1 Version. Added print_atom_list function
#	 add_fragment function
#	 mod_fragment function
#	 fixed bug in min_frag_distance where atom numbers were being output in base 0, instead of base one
#	 filename variable now contains only the filename,not full path, regardless of operating system used
#	 add_dummy_atom function 
#	 get_frag_geometry function
#	 extract_geom_fchk now pulls out the scf energy/total energy 
#Jan 2017: V0pt2 version- NOW works with freq.log files (it takes the input orientation geometry btw)
#	 extract_info_freqlog
#	 property name "fchk_file" now changed to inp_file to reflect greater generality
#	 range of different energy types (scf,ZPE corrected, Gibbs free) added to properties
#	 added create_mol_obj_folder non-class function.
# TO DO: get_element_label() function. When given atom label (e.g 7), return the element it corresponds to.
#April 2017: V0pt3 version - Fixed bug when (in later versions) self.geom = np.zeros((len(geom_list)/3,4)) doesnt work
#	     since its technically a float (though really its an integer value)
#June 2017: V0pt4 version - Now takes the "Standard Orientation" geometry if present in the frequency output file
#			    (otherwise "Input Orientation" is taken)
#June 2017: V0pt5 version - 1) Now takes Charge + Multiplicity from output file for .fchk/.log files
#			    2) Added non-class function "create_gauss_output_mclass" Pass in a list of molecule_class objects
#				and it will save a *.com file with the merged geometry/charge(if present)/multiplicity	
#			    3) Now can parse .com files (gets geom,charge and multiplicity from them-nothing more)
#			    4) Added explicit atomic_number dictionary. keys are atomic symbols, values are atomic numbers

#CURRENT ISSUES:
#1)Wont take the proper energies from MP2 files, will just take scf.
#2)If multiple scf energies are present, it should take the last one before thermochemistry section. Shouldnt be a problem but......
#3)min_point_frag_distance is probably bugged (imagine output atom numbers off by 1). Needs testing if used. BUT adding dummy atom and using min_frag_distance will accomplish anything min_point_frag_distance can do.

class Molecule_Class_V0(object):
	def __init__(self,inp_file):
		self.inp_file = inp_file
		self.charge=None;	#Can probably delete these and just comment them
		self.multiplicity=None;
		self.atom_labels=None;	#Not actually used atm
		self.geom=None;	#nx4 array,1st col is nuclear charge of the atom.col 2-4 = x/y/z co-ords. UNITS=Angstrom
		self.scf_e =None;	#Energy from "SCF Energy" line in .fchk file, "SCF Done" in .log
		self.e_total = None;	#Energy from "Total Energy" line in .fchk file. No equivalent in .log
		self.e_therm_corrd = None	#"Sum of electronic and thermal Energies=" from .log
		self.e_ZPE = None;	#Sum of electronic and zero-point Energies=...
		self.e_enthalpd = None	#Sum of electronic and thermal Enthalpies=
		self.e_gibbs = None	#Sum of electronic and thermal Free Energies=
		self.temp = None	#Temperature usd to get thermal corrections
		self.pressure = None	#Pressure used to get thermal corrections to energy
		assert(inp_file.endswith('.log') or inp_file.endswith('.fchk') or inp_file.endswith('.com')),"Input to Molecule class must be either a .log or .fchk or .com file"	#Only .log and .fchk allowed
		if inp_file.endswith('.fchk'):
			self.extract_geom_fchk(inp_file)
		elif inp_file.endswith('.log'):
			self.extract_info_freqlog(inp_file)
		elif inp_file.endswith('.com'):
			self.parse_gauss_input_file(inp_file)
	#	self.extract_geom_fchk(inp_file) #Extracting geometry(self.geom)
		self.frag_list=["1-" + str(np.shape(self.geom)[0])]	#Each entry contains atoms numbers for a fragment in form (1-4,7,8). FIRST FRAG IS ENTIRE MOLECULE
		#FRAGMENTS IN FRAG_LIST CAN NOT OVERLAP (or at least be REALLY careful if they do)
		self.dist_matrix = None;	#nxn numpy array containing set of distances between all atoms in molecule (see create_dist_matrix function)
	#	self.filename = fchk_file.split('\\')[-1].replace('.fchk','')
		self.filename = os.path.split(inp_file.replace('.fchk',''))[1]
		self.filename = self.filename.replace('.log','')
		self.filename = self.filename.replace('.com','')
	
	def __repr__(self):
		return 'Molecule(%r)'  % (self.inp_file)
	
	
	#Function gets self.geom, self.scf_e, self.e_total
	def extract_geom_fchk(self,fchk_file):
		#1)Pull geom/atom numbers out as string
		with open(fchk_file,'r') as f:
			copy_atom_numb = False #True while copying atom_numb is happening
			copy_geom = False #True while copying geom to string is happening
			charge_found = False #True once the charge has been read from file
			atom_numb_string = ""	
			geom_string = ""
			for line in f:
				#1st if statement for atom_numbers
				if line.find('Atomic numbers')!= -1:
					copy_atom_numb = True
				elif line.find('Nuclear charges')!= -1:
					copy_atom_numb = False
				elif copy_atom_numb:
					atom_numb_string = atom_numb_string + line
				#2nd if state is for geomtry
				if line.find('Current cartesian coordinates')!= -1:
					copy_geom = True
				elif line.find('Force Field')!= -1:
					copy_geom = False	
				elif copy_geom:
					geom_string = geom_string + line
				#3rd/4th if is for SCF E and total E
				if line.find('SCF Energy') != -1:
					self.scf_e = float(line.split()[-1])
				if line.find('Total Energy') != -1:
					self.e_total = float(line.split()[-1])
				#5th/6th = Get the Charge/Multiplicity from file
				if (not charge_found) and (line.find('Charge') != -1):
					self.charge = int( line.strip().split()[-1])
					charge_found = True
				if line.find('Multiplicity') != -1:
					self.multiplicity = int( line.strip().split()[-1]) 
					

#Need to break aat final point
		self.geom_string = geom_string
		#2) Turning the geometry into numpy array
		geom_list = geom_string.split()
		atom_numb_array = atom_numb_string.split()
		self.geom = np.zeros((int(len(geom_list)/3),4))

		for i in range(0,self.geom.shape[0]):	#loop over rows
			self.geom[i,0] = atom_numb_array[i]
			self.geom[i,1] = float(geom_list[i*3]) * 0.52918 #0.52 is to convert to angstrom
			self.geom[i,2] = float(geom_list[(i*3)+1]) * 0.52918
			self.geom[i,3] = float(geom_list[(i*3)+2]) * 0.52918

	#gets self.geom,
	def extract_info_freqlog(self,inp_file):
		#Step 1: Pulling all relevant info from the file into variables
		with open(inp_file,'r') as f:
			copy_geom = True	#True while copying input geometry
			pass_input_orient = False #Turns to true once we pass "Input Orientation" line in file
			geom_list = [] #Each entry is one row in input orientation
			copy_standard_orient = False # True while copying "Standard Otientation" geometry
			for line in f:
			#Charge/Multiplicity
				if line.find("Multiplicity")!=-1:
					self.charge = int(line.strip().split()[2])
					self.multiplicity = int(line.strip().split()[-1])
			#First set of if's extract the geometry
				if line.find("Input orientation")!=-1:
					pass_input_orient = True
				elif pass_input_orient and copy_geom and line.strip().startswith(tuple([str(i) for i in range(0,10)])):
					geom_list.append(line.strip())
				elif pass_input_orient and (line.find("Distance")!=-1 or line.find('Rotational')!=-1):
					copy_geom = False
			#Second set of if statements are for the types of energy
				if line.find('SCF Done') != -1:
					self.scf_e = float(line.strip().split()[4])
				elif line.find('Temperature') != -1:
					self.temp = float(line.strip().split()[1])
					self.pressure = float(line.strip().split()[-2])
				elif line.find('Sum of electronic and zero-point Energies=') != -1:
					self.e_ZPE = float(line.strip().split()[-1])
				elif line.find('Sum of electronic and thermal Energies=') != -1:
					self.e_therm_corrd = float(line.strip().split()[-1])
				elif line.find('Sum of electronic and thermal Enthalpies=') != -1:
					self.e_enthalpd = float(line.strip().split()[-1])
				elif line.find('Sum of electronic and thermal Free Energies=') != -1:
					self.e_gibbs = float(line.strip().split()[-1])

			#Extracts standard_orientation geometry (resets geom_list)
				if line.find("Standard orientation")!=-1:
					copy_standard_orient = True
					geom_list = []
				elif copy_standard_orient and line.strip().startswith(tuple([str(i) for i in range(0,10)])):
					geom_list.append(line.strip())
				elif copy_standard_orient and (line.find("Distance")!=-1 or line.find('Rotational')!=-1):
					copy_standard_orient = False
			
		#Step 2:Putting the extracted geometry into self.geom
		self.geom = np.zeros((int(len(geom_list)),4))
		for row in range(0,self.geom.shape[0]):
			self.geom[row,0] = geom_list[row].split()[1]	#Atomic Number
			self.geom[row,1] = float(geom_list[row].split()[3])	#x co-ord
			self.geom[row,2] = float(geom_list[row].split()[4])	#y co-ord
			self.geom[row,3] = float(geom_list[row].split()[5])	#z co-ord
			

	def parse_gauss_input_file(self,inp_file):	
		""" Extracts charge/geometry/multiplicity when passed a path to gaussian .com file
		
		Args:
			Param1:
				
		Returns
			Nothing but assigns values to self.charge, self.geom and self.multiplicity
		
		Raises:
			Nothing in particular
		"""
		with open(inp_file,'rt') as f:
			full_file=[]
			for line in f:
				full_file.append(line)
			#Find lines with geometry starting/charge on them
			for x in range(0,len(full_file)):
				if full_file[x].find('#') != -1:
					charge_line = x+4
					geom_start = x+5
					break
			#Extract charge,multiplicity and geometry
			self.charge,self.multiplicity = full_file[charge_line].strip().replace(',',' ').split()
			self.charge = int(self.charge)
			self.multiplicity = int(self.multiplicity)
			geom_list = []
			for x in range(geom_start,len(full_file)):
				if full_file[x].strip()!='':
					geom_list.append(full_file[x])
				else:
					break
		
			#Format self.geom correctly
			self.geom = np.zeros((int(len(geom_list)),4))
			for row in range(0,self.geom.shape[0]):
				self.geom[row,0] = atomic_number[geom_list[row].split()[0]]	#Atomic Number
				self.geom[row,1] = float(geom_list[row].split()[1])	#x co-ord
				self.geom[row,2] = float(geom_list[row].split()[2])	#y co-ord
				self.geom[row,3] = float(geom_list[row].split()[3])	#z co-ord





	#returns a numpy array containing the geometry of the fragment
	#Input is the indice of the fragment in frag_list (from that a forammted string of atom numbers is taken
	#Output is nx4 numpy array, same format as self.geom
	def get_frag_geom(self,frag_number):
		atom_numbers = self.frag_list[frag_number] 
		atom_numbers = reformat_list(atom_numbers)
		frag_geom = np.empty((len(atom_numbers),4))	#1st col is atom numbers
		frag_geom[:] = np.nan
		for atom in range(0,len(atom_numbers)):
			curr_atom = int(atom_numbers[atom])
			frag_geom[atom,0] =  self.geom[curr_atom-1,0]
			frag_geom[atom,1] =  self.geom[curr_atom-1,1] #x co-ord
			frag_geom[atom,2] =  self.geom[curr_atom-1,2] #y co-ord
			frag_geom[atom,3] =  self.geom[curr_atom-1,3] #z co-rd
		return frag_geom

	#prints out the atoms which are in the current fragment list
	def print_frag_list(self):
		for fragment in range(0,len(self.frag_list)):
			print('Fragment number', fragment, '=', self.frag_list[fragment])
			

	#Generates a distance (numpy) matrix, which contains all the distances between atoms
	#Currently at least 2x as slow as it could be (its obv a symmetrical matrix)
	def create_dist_matrix(self):
		self.dist_matrix = np.zeros((np.shape(self.geom)[0],np.shape(self.geom)[0]))
		self.dist_matrix[:] = np.nan
		for row in range(0,np.shape(self.dist_matrix)[0]):	
			for col in range(0,np.shape(self.dist_matrix)[0]):
				vector_diff = [self.geom[row,1] - self.geom[col,1],
					       self.geom[row,2] - self.geom[col,2],
					       self.geom[row,3] - self.geom[col,3]]
				self.dist_matrix[row,col] = math.sqrt( (vector_diff[0]**2) + (vector_diff[1]**2) + (vector_diff[2]**2) )

	#Writes *.xyz file in current dir if no folder arg given, otherwise writes in the folder given as a string
	def write_xyz(self,folder=None):
		output_string = str(self.geom.shape[0]) + '\n' + 'Comment line'
		for row in range(self.geom.shape[0]):
			output_string = output_string + '\n' + ('%s %r %r %r' % (atomic_label[self.geom[row,0]],self.geom[row,1],self.geom[row,2],self.geom[row,3]))
		if folder is None:
			folder = os.getcwd()

		with open(os.path.join(folder,self.filename + '.xyz'),'w') as f:
			f.write(output_string)


	#Returns information on the closest atom-atom distance between the 1st fragment and the other fragments
	#OUTPUTS:
	#1) The distance itself (min_inter_frag_distance)
	#2) The number entries in self.geom that these atoms correspond to (atom_numbs)
	#3) The entry number in self.frag_list of the fragment thats closest to the original fragment(the 1st argument) (nearest_frag)
	#INPUTS:
	#1) first_frag = the number of the fragment you want to find the closest contact with. e.g 0 for the first fragment in self.frag_list. CANNOT BE ZERO
	#2) other_frags = LIST of all other fragment numbers.MUST BE A LIST, EVEN IF ONLY ONE FRAGMENT 
	#Defualt is distances between first_frag and all OTHER frags (except frag 0 ) to be calcualted
	def min_frag_distance(self,first_frag,other_frags=None):
		#error checks/defining default values
		if other_frags is None:	#NOTE: the "is" is compltely essentialy, dont replace with diff kind of test.
			other_frags = []
			for fragment in range(1,len(self.frag_list)):
				if fragment != first_frag:
					other_frags.append(fragment)
		if first_frag==0:
			raise SystemExit('first_frag argument in min_frag_distance can not be zero)')
		if self.dist_matrix is None: #Need to generate the distance matrix if it doesnt already exist
			self.create_dist_matrix()
		
		# Create Matrix with each row as an atom in fragment 1, and each column a number in the other fragments (in the order they were passed in) 
		other_frags_atom_list = []	#Get every atom in other_frags as a single entry in list
		other_frag_lengths = [] #Each entry contains the number of atoms in the corresponding fragment in other_frags_atom_list (e.g if 1st other_frag arg is 1-3, then 1st value here will be 3)
		first_frag_list = reformat_list(self.frag_list[first_frag])
		for frag in other_frags:
			other_frags_atom_list += reformat_list(self.frag_list[frag])
			other_frag_lengths.append(len(reformat_list(self.frag_list[frag])))

		#Need to convert lists to base 0(or 1st atom is named "1" and that messes everything up).
		#Output is re-converted into base 1 at the end of the script
		first_frag_list[:] = [x - 1 for x in first_frag_list]
		other_frags_atom_list[:] = [x - 1 for x in other_frags_atom_list]

		#Create the required distance matrix
		frag_dist_matrix = np.zeros((len(first_frag_list),len(other_frags_atom_list)))
		frag_dist_matrix[:] = np.nan
		#Loop should generate correct matrix (DECENT CHANCE OF MISTAKE HERE)	
		for row in range(0,len(first_frag_list)):
			for col in range(0,len(other_frags_atom_list)):
				frag_dist_matrix[row,col] = self.dist_matrix[(first_frag_list[row]),(other_frags_atom_list[col])]
		min_dims = np.unravel_index(frag_dist_matrix.argmin(),frag_dist_matrix.shape) 	#min_dims[0] is the row index, min_dims[1] is the column index of the min value
		
		#Getting the return values
		min_inter_frag_distance = frag_dist_matrix[min_dims]
		atom_numbs = [first_frag_list[min_dims[0]]+1,other_frags_atom_list[min_dims[1]]+1]	#The +1 is beccaues atom numbers were previously converted into base 0 (i.e atom label 1 was labelled 0). This reverses it
		atom_sum = 0
		for entry in range(0,len(other_frag_lengths)):
			atom_sum += other_frag_lengths[entry]
			if min_dims[1] <=atom_sum:
				nearest_frag = other_frags[entry]
				break
		return min_inter_frag_distance,atom_numbs,nearest_frag
		
		#Script finds the shortest distance between a given point (1x3 np array) and the atoms in the fragments passed as args
		#INPUT:
		#1) point_geom = geometry of a point of interest
		#2) other_frags = list of frag numbers (ints needed to access the frags through self.frag_list
		#OUTPUTS:
		#1) The distance itself (min_frag_distance)
		#2) The number entry in self.geom that this atoms correspond to (atom_numb) -  i,e self.geom[atom_numb] will give geometry of that atom)
		#3) The entry number in self.frag_list of the fragment thats closest to the point_geom  (nearest_frag)
		
		#NOTE:May make a TON MORE SENSE to just find the closest atom, can work out any other info from that easily as required
		#NOTE: YET TO BE PROPERLY TESTED. PROBABLY DOESNT WORK PROPERLY
#	def min_point_frag_distance(self,point_geom,other_frags=None):
		#error checks/defining default values
#		if other_frags is None:	#NOTE: the "is" is compltely essentialy, dont replace with diff kind of test.
#			other_frags = [0] # The fragment containing the entire system (i.e ALL atoms)

#		#Getting all the atom numbers in the other_frags + combining
#		other_frags_atom_list = []	#Get every atom in other_frags as a single entry in list
#		other_frag_lengths = [] #Each entry contains the number of atoms in the corresponding fragment in other_frags_atom_list (e.g if 1st other_frag arg is 1-3, then 1st value here will be 3)		
#		for frag in other_frags:
#			other_frags_atom_list += reformat_list(self.frag_list[frag])
#			other_frag_lengths.append(len(reformat_list(self.frag_list[frag])))
#		#Generating a distance matrix
#		point_frag_dist_matrix = np.zeros((len(other_frags_atom_list),1))	#Holds distance between the point_geom and all atoms in all fragments requested	
#		point_frag_dist_matrix[:] = np.nan
#		curr_row = 0	#Tracks where we are in dist matrix
#		for entry in other_frags_atom_list:
#			vector_diff =  [self.geom[entry-1,1] - point_geom[0],	#x
#					self.geom[entry-1,2] - point_geom[1],	#y
#					self.geom[entry-1,3] - point_geom[2]]	#z
#			point_frag_dist_matrix[curr_row,0] = math.sqrt( (vector_diff[0]**2) + (vector_diff[1]**2) + (vector_diff[2]**2) )
#			curr_row +=1
#		#Finding minimum distance
#		min_dims = np.unravel_index(point_frag_dist_matrix.argmin(),point_frag_dist_matrix.shape)
#		atom_numb = other_frags_atom_list[min_dims[0]]	-1 #Number needed to access in the self.geom
#		min_frag_distance = point_frag_dist_matrix[min_dims] 

#		return min_frag_distance,atom_numb

	#Prints a list of atom numbers(1-n) with the element they correspond to
	def print_atom_list(self):
		z=[print(str(x+1) + ' ' + str(  atomic_label[self.geom[x][0]] )) for x in range(0,self.geom.shape[0])]
		#print(z)	

	#Adds a fragmnet to self.frag_list. atoms= list of atoms to for the fragment in format: 2-5,7,13-15
	def add_fragment(self,atoms):
		self.frag_list.append(atoms)
	#Modifies existing fragment number with atom list. fragment = the fragment number (see print_frag_list),atoms = same as add_fragment function
	def mod_fragment(self,fragment,atoms):
		self.frag_list[fragment]=atoms 

	#Adds a dummy atom (symbol X,atomic number -1) with co-ordinates x,y,z to the system
	def add_dummy_atom(self,x,y,z):
		self.geom = np.vstack([self.geom,[-1,x,y,z]])
		self.frag_list[0] = ["1-" + str(np.shape(self.geom)[0])]
		self.dist_matrix = None;	#Need to reset the distance matrix + recalculate with dummy atoms if need be

	#Returns nx4 numpy array with geometry of only the atoms in fragment requested (n= number of atoms in fragment
	#, atomic numbers are in column 1
	# frag number relates to entry in self.frag_list (self.frag_list[frag_number])
	def get_frag_geom(self,frag_number):
		atom_list = reformat_list(self.frag_list[frag_number]) 
		frag_geom = np.zeros((len(atom_list),4))
		counter = 0
		print(atom_list)
		for atom in atom_list:
			frag_geom[counter] = self.geom[atom-1] #Atoms are base 1, list indices base 0. corrects for that
			counter = counter + 1
		return frag_geom

#NON CLASS FUNCTIONS START HERE


#Turns a list in format 1-5,7,9 into a list of ints (e.g 1,2,3,4,5,7,9)
#number_string = a STRING with commas/dashes and numbers and NOTHING ELSE
#output_list = a list of numbers
def reformat_list(number_string):
	number_string = number_string.split(',')
	output_list =[] 
	for part in number_string:
		if '-' in part:
			a,b = part.split('-')
			a,b = int(a), int(b)
			output_list.extend(range(a,b+1))
		else:
			a = int(part)
			output_list.append(a)
	return output_list

#Returns the co-ordinates of the center of mass of input geometry
#geometry = nx4 (4 cols) numpy array, col 1 = atom NUMBERS, col 2/3/4=x/y/z co-ords
#center_mass = 1x3 array(NOT NUMPY) containg x/y/z center of mass
def find_center_mass(geometry):
	curr_sum = [0,0,0] #holds x/y/z COM
	mass_sum = 0	#tracks sum of all masses
	for atom in geometry:
		curr_mass = atomic_mass[atomic_label[atom[0]]]
		mass_sum += curr_mass
		curr_sum[0] = curr_sum[0] + (atom[1] * curr_mass)	#x vals 
		curr_sum[1] = curr_sum[1] + (atom[2] * curr_mass)	#y vals
		curr_sum[2] = curr_sum[2] + (atom[3] * curr_mass)	#z vals
	center_mass = [x / mass_sum for x in curr_sum]
	return center_mass

#Returns the co-ordinates of the center of the input geometry
#geometry = nx3 (3 cols) numpy array, col 1/2/3=x/y/z co-ords
#center_geom = 1x3 array(NOT NUMPY) containg x/y/z center of mass
def find_center_geom(geometry):
	curr_sum = [0,0,0]
	for atom in geometry:
		curr_sum[0] = curr_sum[0] + atom[0] 	#x vals 
		curr_sum[1] = curr_sum[1] + atom[1] 	#y vals
		curr_sum[2] = curr_sum[2] + atom[2] 	#z vals
	center_geom = [x / np.shape(geometry)[0] for x in curr_sum]
	return center_geom
	
#returns list of molecule objects, 1 for each relevant file found in "folder"
#Exits with error if atomic numbering is found to be incosistent between files
#INPUT:
#folder = full path to the folder containing the files of interest 
#ext = the extension of the files of interest (either '.log' or '.fchk' 
def create_mol_obj_folder(folder,ext):
	error=0 #If this becomes 1 at some point an error gets thrown
	#----->Create the list of molecule objects for the folder
	file_list = [os.path.join(folder,file) for file in os.listdir(folder) #List of paths for all "ext" files in folder
		     if os.path.isfile(os.path.join(folder,file)) #Files only (i.e not dirs)
		     and file.endswith(ext)]	#Only files with required extension
	mol_list = [Molecule_Class_V0(file) for file in file_list]	#Actual list of objects
	#----->Checking same number of atoms in all files (Error if there isnt):
	numb_atoms = [molecule.geom.shape[0] for molecule in mol_list]	#Number of atoms all files
	for entry in numb_atoms:
		if numb_atoms[0]!=entry:
			error=0
			print('Folder:',folder)
			[print('File:',molecule.filename,',Number atoms:',molecule.geom.shape[0]) for molecule in mol_list]
			assert error==0, "ERROR: Number of atoms not equal in all files in folder " + folder 
	#----> Checking all atom lists have elements in same order (imperfect way of testing the atomic nubmering is constant
	all_atoms = np.stack([molecule.geom[:,0] for molecule in mol_list])	#Each row is a complete list of atomic numbers
	if not(((all_atoms - all_atoms[0]==0)).all()):	#Subtract 1st row off all rows and see if zero
		error=1
		indices = np.transpose(np.nonzero(all_atoms - all_atoms[0])) #1st col= indices of rows in all_atoms where numbering different to row 1
		indicies = indices[:,0]	#Only care about first col(see comment above)
		mismatches = [mol_list[molecule].filename for molecule in range (0,len(mol_list))
				if np.any(indices==molecule)]	#Filenames for files not matching numbering in 1st file
		[print('File:',mol_list[0].filename,'atomic numbering !=', entry,'\n') for entry in mismatches]
		assert error==0,"Atomic Numbering is not consistent for files in folder:" + folder

	return mol_list	






def create_gauss_output_mclass(*mol_objs,**kwargs):
	""" takes n mol-objects as positional arguments, merges geometries and saves a gaussian .com output file
	
	args:
		mol_objs: Each argument is a Molecule_Class_V0 object
	
	keyword Args(All optional):
		charge: int, Charge for the ouput file (default is sum(charge) from mol_objs*)
		multiplicity: int, Multiplicity for output file, defaults to maximising number of unpaired electrons. 
		gauss_line: str, Route section for gaussian 09 (line begining with # and containing functional/basis set
		extra_line: str, line to write after geometry (e.g smd parameters). No need to put blank line @ end.
		output_fname:str, name of output file
		output_folder:str, path to FOLDER to save output file in				
	returns
		Saves a .com file with merged geometry
	
	raises:
		None at the mo. laziness.
	"""
	molecules = [ Molecule_Class_V0(x) for x in mol_objs] 

	#Figuring out/defining default arguments:
	unpair_e = sum([(x.multiplicity -1) for x in molecules])
	multiplicity = kwargs.get('multiplicity',unpair_e+1)
	charge = kwargs.get('charge',sum([x.charge for x in molecules]))
	gauss_line = kwargs.get('gauss_line','# B3LYP/6-311+G(d,p) int=ultrafine scf=conver=9 empiricaldispersion=gd3bj nosymm')
	extra_line = kwargs.get('extra_line',None)
	output_fname = kwargs.get('output_fname',molecules[0].filename + '.com')
	output_folder = kwargs.get('output_folder',os.path.split(molecules[0].inp_file)[0])

	#Merging geometires
	if len(molecules)>1:
		output_geom = np.concatenate( tuple( (x.geom for x in molecules) ) )
	else:
		output_geom = molecules[0].geom

	#Now writing the output file:
	output_str = [] 
	output_str.extend(['%nprocshared=4','%mem=7500MB','%chk=' + molecules[0].filename + '.chk'])
	output_str.append(gauss_line)
	output_str.extend([' ','TITLE',' '])
	output_str.append(str(charge) + ',' + str(multiplicity))
	for row in range(0,len(output_geom)):
		temp_list = output_geom[row].tolist()
		temp_list[0] = atomic_label[int(temp_list[0])]
		temp_str = "{:3}    {:.6f}    {:.6f}    {:.6f}".format(*temp_list)
		temp_str = temp_str.replace(' -','-')
		output_str.append(temp_str)
	output_str.append(' ')
	if extra_line is not None:
		output_str.append(extra_line)
		output_str.append(' ')
	with open (os.path.join(output_folder,output_fname),'wt') as f:
		[f.write(x+'\n') for x in output_str]
	




#DICTIONARIES START HERE


atomic_mass = dict(X=0.00,H=1.01, He=4.00, Li=6.94, Be=9.01, B=10.81, C=12.01,
                   N=14.01, O=16.00, F=19.00, Ne=20.18, Na=22.99, Mg=24.31,
                   Al=26.98, Si=28.09, P=30.97, S=32.07, Cl=35.45, Ar=39.95,
                   K=39.10, Ca=40.08, Sc=44.96, Ti=47.87, V=50.94, Cr=52.00,
                   Mn=54.94, Fe=55.85, Co=58.93, Ni=58.69, Cu=63.55, Zn=65.39,
                   Ga=69.72, Ge=72.61, As=74.92, Se=78.96, Br=79.90, Kr=83.80,
                   Rb=85.47, Sr=87.62, Y=88.91, Zr=91.22, Nb=92.91, Mo=95.94,
                   Tc=98.00, Ru=101.07, Rh=102.91, Pd=106.42, Ag=107.87,
                   Cd=112.41, In=114.82, Sn=118.71, Sb=121.76, Te=127.60,
                   I=126.90, Xe=131.29, Cs=132.91, Ba=137.33, La=138.91,
                   Ce=140.12, Pr=140.91, Nd=144.24, Pm=145.00, Sm=150.36,
                   Eu=151.96, Gd=157.25, Tb=158.93, Dy=162.50, Ho=164.93,
                   Er=167.26, Tm=168.93, Yb=173.04, Lu=174.97, Hf=178.49,
                   Ta=180.95, W=183.84, Re=186.21, Os=190.23, Ir=192.22,
                   Pt=195.08, Au=196.97, Hg=200.59, Tl=204.38, Pb=207.2,
                   Bi=208.98, Po=209.00, At=210.00, Rn=222.00, Fr=223.00,
                   Ra=226.00, Ac=227.00, Th=232.04, Pa=231.04, U=238.03,
                   Np=237.00, Pu=244.00, Am=243.00, Cm=247.00, Bk=247.00,
                   Cf=251.00, Es=252.00, Fm=257.00, Md=258.00, No=259.00,
                   Lr=262.00, Rf=261.00, Db=262.00, Sg=266.00, Bh=264.00,
                   Hs=269.00, Mt=268.00)
atomic_label = {-1:'X',1:'H', 2:'He', 3:'Li', 4:'Be', 5:'B', 6:'C',
		7:'N', 8:'O', 9:'F', 10:'Ne', 11:'Na', 12:'Mg',
		13:'Al', 14:'Si', 15:'P', 16:'S', 17:'Cl', 18:'Ar',
		19:'K', 20:'Ca', 21:'Sc', 22:'Ti', 23:'V', 24:'Cr',
		25:'Mn', 26:'Fe', 27:'Co', 28:'Ni', 29:'Cu', 30:'Zn',
		31:'Ga', 32:'Ge', 33:'As', 34:'Se', 35:'Br', 36:'Kr',
		37:'Rb', 38:'Sr', 39:'Y', 40:'Zr', 41:'Nb',42:'Mo',
		43:'Tc', 44:'Ru', 45:'Rh', 46:'Pd', 47:'Ag', 48:'Cd',
		49:'In', 50:'Sn', 51:'Sb', 52:'Te', 53:'I', 54:'Xe',
		55:'Cs', 56:'Ba', 57:'La', 58:'Ce', 59:'Pr', 60:'Nd',
		61:'Pm', 62:'Sm', 63:'Eu', 64:'Gd', 65:'Tb', 66:'Dy',
		67:'Ho', 68:'Er', 69:'Tm', 70:'Yb', 71:'Lu', 72:'Hf',
		73:'Ta', 74:'W', 75:'Re', 76:'Os', 77:'Ir', 78:'Pt',
		79:'Au', 80:'Hg', 81:'Tl', 82:'Pb', 83:'Bi', 84:'Po',
		85:'At', 86:'Rn', 87:'Fr', 88:'Ra'}
		
atomic_number = {v:k for k,v in atomic_label.items()}


