#!/usr/bin/perl

use lib "/home/lmt09/PHD_Y3/PROGRAMS/PERL/gregap/";
use ESPT::Glog 0.07;
use File::Basename;

# When adding new options you need to update:
# 1. SYNOPSIS manpage
# 2. Argument check (default set to zero and if check section)
# 3. OPTIONS manpage
# 4. Input error if no input
# 5. main while loop and data output
# 6. call to plotopt subroutine
# 7. usage subroutine

# Things still to add:
# 1. add oniom functionality
# 2. add normalization for Oniom comparison

=head1 NAME

GREGAP - Gaussian Regular Expression Graphical Analysis in Perl 

=head1 SYNOPSIS

B<gregap>  [ B<-d> F<Gaussian_log_file> ] [ B<-e> root(s) F<Gaussian_log_file> ] [ B<-f> F<Gaussian_log_file> ] [ B<-o> layer(s) F<Gaussian_log_file> ] [ B<-h> ] 

=head1 DESCRIPTION

This program uses GNUPlot to allow graphical monitoring of Gaussian jobs.

=cut

### Main Program ###
our $version = "1.1";

#######################################################################
# GREGAP                                                              #
# CREATED 05/03/11                                                    #
# LAST MODIFIED 15/03/11                                              #
# LEE THOMPSON                                                        #
# This Program takes the forces from each geometry optimization step  #
# and shows it in graphical format for debugging purpose. Future      #
# intention to extend this further with options.                      #
#######################################################################

# check x11 in operation
$display = $ENV{'DISPLAY'};
if ($display =~ /^$/){
	die "\n\tX11 not set, check your DISPLAY variable\n\n";
}

# check for arguments 
usage() if ( $#ARGV < 0 );

help() if $ARGV[0] eq "-h"; 
help() if $ARGV[0] eq "--help";

$haveInput = 0;
$isLogFile = 0;
$DoForces = 0;
$DoDisp = 0;
$DoEnerg = 0;
$DoOniom = 0;

for (my $i=0; $i<=$#ARGV; $i++) { 
	if ($ARGV[$i] eq "-f"){
		$infile = $ARGV[$i+1];
		$haveInput = 1;
		$DoForces = 1;
	}
	if ($ARGV[$i] eq "-d"){
		$infile = $ARGV[$i+1];
		$haveInput = 1;
		$DoDisp = 1;
	}
        if ($ARGV[$i] eq "-e"){
		if (-e $ARGV[$i+1]){
			$nroot[0] = 1;
			$infile = $ARGV[$i+1];
			$haveInput = 1;
			$DoEnerg = 1;
		}else{
			for($j=$i+1;$j<=$#ARGV;$j++){
				unless (-e $ARGV[$j]){
					$nroot[$n++] = $ARGV[$j];
				}else{
					$infile = $ARGV[$j];
					$haveInput = 1;
					$DoEnerg = 1;
				}
			}
		}
		if (scalar @nroot > 1){
			$multilevs = scalar @nroot;
		}else{
			$multilevs = 0;
		}
	} 
	if ($ARGV[$i] eq "-o"){
		if (-e $ARGV[$i+1]){
			$nlayer[0] = "tot";
			$infile = $ARGV[$i+1];
			$haveInput = 1;	
			$DoOniom = 1;
		}else{
			for($j=$i+1;$j<=$#ARGV;$j++){
				unless (-e $ARGV[$j]){
					unless ($ARGV[$j] eq "tot" || $ARGV[$j] eq "lm" || $ARGV[$j] eq "hm" || $ARGV[$j] eq "lr"){
						die "ERROR: valid arguments to -o are \"tot\", \"lm\", \"hm\" or \"lr\"\n";
					}
					$nlayer[$n++] = $ARGV[$j];
				}else{
					$infile = $ARGV[$j];
					$haveInput = 1;
					$DoOniom = 1;
				}
			}
		}
		if (scalar @nlayer > 1){
			$multilayer = scalar @nlayer;
		}else{
			$multilayer = 0;
		}	
	}
}

=head1 OPTIONS

Command line option specifications are processed from left to right and may 
be specified more than once. If conflicting options are specified, later ones take precedence.

=over 16

=item B<-d>

Display RMS and maximum displacement against iteration number.

=item B<-e> root(s)

Display Energy in Hartree against iteration number of specified roots. If more than one root is specifies displays all of them. If no root is specifies defaults to 1st root.

=item B<-f>	

Display RMS and maximum force against iteration number.

=item B<-o> layer(s)

Display Energy of specified ONIOM layer against iteration number. If more than one layer is specified displays all of them. If no root is specified defaults to extrapolated energy. Note that the energy scales of different layers may be very different. Acceptable input options are "lm", "hm", "lr" or "tot"

=item B<-h>

=item B<--help> 	

Print full GREGAP documentation via perldoc. Cannot be used with other options.

=back

=cut

print "\nGREGAP $version : Gaussian RegEx Graphical Analysis \n\n";

if ($haveInput == 0){
	if (@ARGV[0] =~ /^-[def]/){
		print "Gaussian log file missing.\n";
		die "Exit.\n $!\n";
	}elsif (@ARGV[0] =~ /^-/){
		print "### Invalid option ###\n";
		usage();
	}else{
		print "### No option specified ###\n";
		usage();
	}
}

# If logfile is older than plot.dat then no need to re-read.
# Energies should always be re-read as no way to check if roots are different
if ((stat("$infile"))[9] > (stat('plot.dat'))[9] ||  $DoEnerg == 1) {

open(INFILE,$infile) or die "Could not read $infile\n$!\n";

$[ = 1;			# set array base to 1

open(DATA,'>plot.dat') || die "Cannot open plot data file!\n";

my $Glog = ESPT::Glog->new();
$Glog->analyze($infile);
$route = $Glog->get(ROUTE);
#print "$route\n";
#$ftype = $Glog->get(JOBTYPE);
#print "$ftype\n";
#@keywords[$i++] = $Glog->get(KEYWORDS);
#foreach (@keywords){
#	print "$_\n";
#}
#$fcomp = $Glog->get(COMPLETE);
#print "$fcomp\n";
#@eelec = $Glog->get(EELEC);
#foreach (@eelec){
#	print "$_\n";
#}

#print "$route\n";

if ($route =~ /oniom/i && $DoOniom == 0 && $DoEnerg == 1){
	print "WARNING: ONIOM keyword detected in route\n";
	print "\tuse of energy with oniom may give unpredicatable results";
}

while(<INFILE>){
    next if /^$/;

# Check input is a Gaussian log file
    if ($isLogFile == 0){
	if( /^\s+Entering\s+Gaussian\s+System/ ){
		$isLogFile = 1;
		print "Input file $infile is a Gaussian log file.\n";
    	}else{
		print "Input file $infile is not a Gaussian log file.\n";
		die "Exit.\n\n";
    	}
    }

# Search for regular expression, extract values to arrays
    if (/Maximum Force\s+([\d\.]+)/) {
    	push @MF, $1;
    }
    if (/RMS     Force\s+([\d\.]+)/) {
    	push @RF, $1;
    }
    if (/Maximum Displacement\s+([\d\.]+)/){
	push @MD, $1;
    }
    if (/RMS     Displacement\s+([\d\.]+)/){
        push @RD, $1;
    }
    foreach $root (@nroot){ 
    	$/ = "";
    	if ($multilevs > 1){
    		$EVarNam = "EEl" . "$root";
    	}else{
		$EVarNam = "EEl1";
    	}
    	if ($route =~ /cas/i && /Do an extra-iteration for final printing.*\(  $root\)\s+Eigenvalue\s+(-*\d+\.\d+)/s){
		push @$EVarNam, $1;
    	}elsif (/\( $root\)\s+EIGENVALUE\s+(-*\d+\.\d+)/){
		push @$EVarNam, $1;
	}elsif (/SCF\s+Done:\s+[\w\)\(]+\s+=\s+(-*\d+\.\d+)/){
		push @EEl1, $1;
    	}
    }
    if (/ONIOM: extrapolated energy =\s+(-*\d+\.\d+)/){
		push @ONItot, $1;
    }
    if (/ONIOM: gridpoint  1 method:  low   system:  model energy:\s+(-*\d+\.\d+)/){ 	
		push @ONIlm, $1;
    }
    if (/ONIOM: gridpoint  2 method:  high  system:  model energy:\s+(-*\d+\.\d+)/){
   		push @ONIhm, $1;
    }
    if (/ONIOM: gridpoint  3 method:  low   system:  real  energy:\s+(-*\d+\.\d+)/){
		push @ONIlr, $1;
    }
}

# Output extracted data in plot.dat
print DATA "# ITN\tRMS Forc\tMax Forc\tRMS Disp\tMax Disp\tSCF Ener1\tSCF Ener2\tSCF Ener3\tSCF Ener4\tSCF Ener5\tONIOM tot\tONIOM lm\t ONIOM hm\tONIOMlr\n";
for ($i = 1; $i <= scalar @MF; $i++) {
	printf DATA ("%d\t%f\t%f\t%f\t%f\t%f\t%f\t%f\t%f\t%f\t%f\t%f\t%f\t%f\n", $i, $RF[$i], $MF[$i], $RD[$i], $MD[$i], $EEl1[$i], $EEl2[$i], $EEl3[$i], $EEl4[$i], $EEl5[$i], $ONItot[$i], $ONIlm[$i], $ONIhm[$i], $ONIlr[$i]);
}

close DATA;
close INFILE;

}else{
	print "log file is older than plot.dat\n\tSkipping log file read...\n\nDelete plot.dat to re-read\n";
}

# Call plotopt to plot the data in plot.dat
#
if ($DoForces == 1){
	&plotopt('plot.dat', 'Iteration', 'Force', 'CONVERGENCE', '', '', 'lines','2|3', 'RMS Force|Max Force');
}
if ($DoDisp == 1){
	&plotopt('plot.dat', 'Iteration', 'Displacement', 'CONVERGENCE', '', '', 'lines', '4|5', 'RMS Displacement|Max Displacement');
}

if ($DoEnerg == 1 && $multilevs == 2){
	&plotopt('plot.dat', 'Iteration', 'Energy (Har)', 'ENERGY', '', '', 'lines', "6|7", "Root $nroot[0]|Root $nroot[1]");
}elsif ($DoEnerg == 1 && $multilevs == 3){
        &plotopt('plot.dat', 'Iteration', 'Energy (Har)', 'ENERGY', '', '', 'lines', '6|7|8', "Root $nroot[0]|Root $nroot[1]|Root $nroot[2]");
}elsif ($DoEnerg == 1 && $multilevs == 4){
        &plotopt('plot.dat', 'Iteration', 'Energy (Har)', 'ENERGY', '', '', 'lines', '6|7|8|9', "Root $nroot[0]|Root $nroot[1]|Root $nroot[2]|Root $nroot[3]");
}elsif ($DoEnerg == 1 && $multilevs == 5){ 
        &plotopt('plot.dat', 'Iteration', 'Energy (Har)', 'ENERGY', '', '', 'lines', '6|7|8|9|10', "Root $nroot[0]|Root $nroot[1]|Root $nroot[2]|Root $nroot[3]|Root $nroot[4]"); 
}elsif ($DoEnerg == 1 && $multilevs == 0){
	&plotopt('plot.dat', 'Iteration', 'Energy (Har)', 'ENERGY', '', '', 'lines', '6', '');
}

if ($DoOniom == 1){
	if ($nlayer[0] eq "tot"){ 
		$onicol = 11;
	}elsif ($nlayer[0] eq "lm"){
		$onicol = 12;
	}elsif ($nlayer[0] eq "hm"){
		$onicol = 13;
	}elsif ($nlayer[0] eq "lr"){
		$onicol = 14;
	}
 	if ($multilayer == 0){
		&plotopt('plot.dat', 'Iteration', 'Energy (Har)', 'ENERGY', '', '', 'lines', "$onicol", "ONIOM $nlayer[0]");
	}else{
		print "multilayer ONIOM analysis not supported yet!\n"
	}
}

    #######################################################################
    # SUBROUTINE PLOTOPT                                                  #
    # CREATED 10/05/12                                                    #
    # LAST MODIFIED 10/05/12                                              #
    # LEE THOMPSON                                                        #
    # This subroutine takes data from plot.dat, constructs a gnuplot file #
    # plot.plt and displays the gnuplot graphic. Series and columns input #
    # use "|" as a delimiter. First column is plotted as x data and each  #
    # subsequent column corresponds to y data for each series. Extend by  #
    # inserting option for surface plot.                                  #
    #######################################################################

sub plotopt {
    local($input, $xlabel, $ylabel, $title, $xrange, $yrange, $style, $column,$series) = @_;

    $[ = 1;

    open FHDL, ">plot.plt" or die "Cannot open plot input\n";

    @names = split(/\|/, $series);
    @columns = split(/\|/, $column);
    if (scalar @columns > 1) {
	$command = sprintf("plot \"%s\" using 1:%s title \"%s\"\n", $input,$columns[1], $names[1]);
	for ($i = 2; $i <= scalar @columns; $i++) {
	    $command = sprintf("%sreplot \"%s\" using 1:%s title \"%s\"\n",$command, $input, $columns[$i], $names[$i]);
	}
    }
    elsif (scalar @names == $column) {
	$command = sprintf("plot \"%s\" using 1 title \"%s\"\n", $input,$names[1]);
	for ($i = 2; $i <= $column; $i++) {
	    $command = sprintf("%sreplot \"%s\" using %s title \"%s\"\n",$command, $input, $i, $names[$i]);
	}
    }
    else {
	$command= sprintf("plot \"%s\" using 1:%s title \"%s\"\n", $input,$column, $series);
    }

    printf FHDL "set xlabel \"%s\"\n" . "set ylabel \"%s\"\n" . "set xrange [%s]\n" . "set yrange [%s]\n" . "set grid\n" .  "set data style %s\n" . "set title \"%s\"\n" . '%s' . 'pause -1', $xlabel, $ylabel, $xrange, $yrange, $style, $title, $command;
    close FHDL;

    system('gnuplot plot.plt');
}


    #######################################################################
    # SUBROUTINE HELP                                                     #
    # CREATED 15/05/12                                                    #
    # LAST MODIFIED 10/05/12                                              #
    # LEE THOMPSON                                                        #
    # Full help documentation                                             #
    #######################################################################

sub help {
	system("perldoc /home/lmt09/PHD_Y3/PROGRAMS/PERL/gregap/gregap.pl");
	exit;
}


    #######################################################################
    # SUBROUTINE USAGE                                                    #
    # CREATED 15/05/12                                                    #
    # LAST MODIFIED 10/05/12                                              #
    # LEE THOMPSON                                                        #
    # Brief usage documentation                                           #
    #######################################################################

sub usage {
	print "\nGREGAP $version : Gaussian Regular Expression Graphical Analysis in Perl \n";
	print "\nUsage: gregap [options] <Gaussian_logfile>\n";
	print "\t-d\t\t\tdisplay RMS and Maximum displacement\n";
	print "\t-e\t\t\tdisplay energy per iteration\n";
	print "\t-f\t\t\tdisplay RMS and Maximum force\n";
	print "\t-o\t\t\tdisplay energy of ONIOM layer per iteration\n";
	print "\t-h\t\t\tprint full documentation\n"; 
	print "\n";
	exit;
}

=head1 EXAMPLES

=over

=item gregap

Called without any parameters, GREGAP will display usage information.
If B<-h> or B<--help> is passed, then the full GREGAP documentation is displayed via perldoc.

=item gregap -f foo.log

GREGAP reads foo.log and creates plot of Max Force and RMS Force.

=back

=head1 NOTES

GREGAP

=head1 VERSION

1.1

=head1 AUTHOR

Lee Thompson, E<lt>l.thompson09@imperial.ac.ukE<gt>

=head1 COPYRIGHT

Copyright (c) 2012 Lee Thompson

This is free software; modification and redistribution is made under the same terms as Perl itself.

=cut

