#!/usr/bin/python
################################################################################
#
# Collection of functions for reading data from Gaussian formatted checkpoint
# files and for writing data back in the same format.
#
#  by Fernando R. Clemente (Gaussian, Inc.)
#
#  Last updated: December 2013
#
################################################################################
#
# function that stores the formats used in fchk arrays
def fchk_aformats(AType):
    """
    Given an array type it returns a tuple with the number of values per line,
    the length of each values, the format in which values should be printed to
    conform with a Gaussian formatted checkpoint file, and the data type.
    Last value in the tuple if the format in which the value should be printed
    if item is scalar instead of an array.
    """
    FFormats = { 'I' : (6, 12,   '12d',   int,    '12d'),
                 'R' : (5, 16, '16.8E', float, '22.15E'),
                 'C' : (5, 12,   '12s',   str,    '12s'),
                 'H' : (9,  8,    '8s',   str,       ''),
                 'L' : (0,  0,      '',   str,     '1s') }
    return FFormats[AType]
#
# function that returns info for a given item in a fchk file
def item_info(LString):
    """
    Given a string (a line in a fchk file where an item is given), it returns
    the title of the item and a dictionary with:
        - "Type" = the type (R,I,C,...),
        - "IsArray" = an empty string if item is a scalar,
        - "Value" = scalar value if item is a scalar, or the number of elements
          in the array if item is an array.
    """
    Title = LString[0:40].rstrip()
    Type = LString[43]
    IsArray = LString[47:49].lstrip()
    ItemInfo = { "Type" : Type , "IsArray" : IsArray }
    if IsArray:
        Value = int(LString[49:])
    else:
        Value = fchk_aformats(Type)[3](LString[49:])
    ItemInfo["Value"] = Value
    return Title, ItemInfo
#
# function that returns the number of lines in an array
def array_nlines(NVals, AType):
    """
    Total number of lines in an array given the total number of elements and
    the array type.
    """
    PerLine = fchk_aformats(AType)[0]
    if NVals % PerLine:
        return NVals // PerLine + 1
    else:
        return NVals / PerLine
#
# function that extracts one item from a fchk file
def fchk_item(FChkFile, ItemTitle):
    """
    Extracts one specific item from a formatted checkpoint file given the name
    of a Gaussian formatted checkpoint file and the item title.
    """
    with open(FChkFile) as FChk:
        ReadArray = 0
        LineID = 0
        for Line in FChk:
            Line = Line.rstrip('\r\n')
            LineID += 1
            if ItemTitle in Line:
                Title, Item = item_info(Line)
                if Item["IsArray"]:
                    NLines = array_nlines(Item["Value"], Item["Type"])
                    ReadArray = 1
                    EndArray = LineID + NLines
                    Array = []
                    VLen = fchk_aformats(Item["Type"])[1]
                    VType = fchk_aformats(Item["Type"])[3]
            elif ReadArray:
                NewEles = [ VType(Line[VSt : VSt + VLen]) for VSt in range(0,
                    len(Line), VLen) ]
                Array.extend(NewEles)
                if LineID == EndArray:
                    Item["Array"] = Array
                    Item["Value"] = len(Item["Array"])
                    ReadArray = 0
    return Title, Item
#
# function that loads the entire fchk file
def fchk_all(FChkFile):
    """
    Reads an entire formatted checkpoint file and returns a dictionary that
    contains every item in the checkpoint file. The keys are the item names
    (Titles) and the values are itself dictionaries with:
        - "Type" = the type (R,I,C,...),
        - "IsArray" = an empty string if item is a scalar,
        - "Value" = scalar value if item is a scalar, or the number of elements
          in the array if item is an array,
        - "Array" = a list with the array values if item is an array (this key
          is not present if item is a scalar).
    "Line 1" and "Line 2" are two additional items in the FChkD dictionary that
    is returned, which contain the first two lines in the fchk file as strings.
    The function also returns a list with the names of the items found, so the
    fchk file can be printed back with the items appearing in the same order,
    if needed.
    """
    FChkD = {}
    FChkTitles = []
    with open(FChkFile) as FChk:
        ReadArray = 0
        LineID = 0
        for Line in FChk:
            Line = Line.rstrip('\r\n')
            LineID += 1
            if LineID <= 2:
                Title = "Line {0:d}".format(LineID)
                FChkD[Title] = Line
                FChkTitles.append(Title)
            elif ReadArray:
		NewEles = []
                for VSt in range(0, len(Line), VLen):
                    try:
                        NewEles.append(VType(Line[VSt : VSt + VLen]))
                    except ValueError:
                        NewEles.append(VType(0))
                Array.extend(NewEles)
                if LineID == EndArray:
                    Item["Array"] = Array
                    Item["Value"] = len(Item["Array"])
                    ReadArray = 0
                    FChkD[Title] = Item
            else:
                Title, Item = item_info(Line)
                if Item["IsArray"]:
                    NLines = array_nlines(Item["Value"], Item["Type"])
                    ReadArray = 1
                    EndArray = LineID + NLines
                    Array = []
                    VLen = fchk_aformats(Item["Type"])[1]
                    VType = fchk_aformats(Item["Type"])[3]
                else:
                    FChkD[Title] = Item
                FChkTitles.append(Title)
    return FChkD, FChkTitles
#
# function that returns the string representation of a fchk array for printing
def prt_fchk_array(AName, AType, VList):
    """
    Returns a string in the format of an array in a Gaussian formatted
    checkpoint file. It takes a list of values (the array), an array name that
    will appear in the FChk file and the array type (I, R, C, H for integer,
    real, and character, two types).
    """
    NVal = len(VList)
    FHeader = '{0:40s}   {1:1s}   N={2:12d}\n'.format(AName, AType, NVal)
    (VPLine, VLen) = fchk_aformats(AType)[0:2]
    VFormat = '{0:' + fchk_aformats(AType)[2] + '}'
    FList = []
    for VSt in range(0, NVal, VPLine):
        FList.extend([ VFormat.format(__) for __ in VList[VSt : VSt + VPLine] ]
                + ['\n'])
    return FHeader + ''.join(FList)
#
# function that returns the string representation of a fchk scalar for printing
def prt_fchk_scalar(SName, SType, SVal):
    """
    Returns a string in the format of a scalar in a Gaussian formatted
    checkpoint file.
    """
    SFormat = '{0:40s}   {1:1s}     {2:' + fchk_aformats(SType)[4] + '}\n'
    FChkSc = SFormat.format(SName, SType, SVal)
    return FChkSc
#
# function that prints a complete fchk
def prt_fchk(FChkD, LTitles):
    """
    Returns the string representation of a complete Gaussian formatted
    checkpoint file given a dictionary with all the items in the fchk file plus
    a list with the titles of the items.
    """
    FList = [ FChkD['Line 1'], '\n', FChkD['Line 2'], '\n' ]
    for Title in LTitles[2:]:
        Item = FChkD[Title]
        if Item['IsArray']:
            FList.extend(prt_fchk_array(Title, Item['Type'], Item['Array']))
        else:
            FList.extend(prt_fchk_scalar(Title, Item['Type'], Item['Value']))
    return ''.join(FList)

####################
#
# Do nothing if called as a script; the functions should just be imported as
# modules.
#
if __name__ == "__main__":
    pass

