#!/usr/bin/python
#
# Modify the first line if your python interpreter is not
# in /usr/bin, so it points to the correct location.
# This has been tested with at least Python 2.6, it may not
# work without modification in older versions of Python.
#
################################################################################
'''
          "irc_merge.py" by Fernando R. Clemente (Gaussian, Inc.)
                               March 2014

    This program merges the IRC data from two different Gaussian formatted
    checkpoint files, in principle, one corresponding with an IRC run on the
    "forward" direction and the other one corresponding with an IRC run on the
    "reverse" direction. The Gaussian formatted checkpoint file for the job in
    the "forward" direction is used as reference template for the combination
    and the IRC data present in the file for the "reverse" direction is added
    to it. The program will also delete the duplicate entry for the TS since
    this point would be present in the formatted checkpoint files for both
    "forward" and "reverse" directions. The results of the combination are
    saved as a new Gaussian formatted checkpoint file that can be visualized in
    GaussView.

    ---------------------------------------------------------------------------

    Usage:
            irc_merge.py  IRC-example1.fchk  IRC-example2.fchk  IRC-combined.fchk


            The first two names must be the names of the two currently existing
            Gaussian formatted checkpoint files with the IRC datat that one
            would like to merge. The order in which these two files are given
            does not matter.

            The third argument must be the name of the Gaussian formatted
            checkpoint file where the merged data will be saved.

    ---------------------------------------------------------------------------
'''
#
# Import required modules
import sys
#
# It uses a few modules from "readfchk.py", so make sure that such file is on
# the same directory as the program here or that "readfchk.py" is in a
# directory in your PYTHONPATH.
from readfchk import fchk_all, prt_fchk
#
# Process command line arguments: names of three Gaussian fchk files.
# Print help message and quit if incorrect number of arguments or "-h" switch.
try:
    if sys.argv[1] == "-h":
        print __doc__
        quit()
    if len(sys.argv) != 4:
        raise IndexError
except IndexError:
    print __doc__
    quit()
#
# Read the two formatted checkpoint files with IRC data to be merged.
#
FChkA, TitlesA = fchk_all(sys.argv[1])
FChkB, TitlesB = fchk_all(sys.argv[2])
#
# IRC arrays that are to be merged.
#
IRCArrays = [ ('IRC point       1 Results for each geome', 'IRC Num results per geometry'),
              ('IRC point       1 Geometries', 'IRC Num geometry variables'),
              ('IRC point       1 Gradient at each geome', 'IRC Num geometry variables') ]
#
# The reaction coordinate values in the "Results for each geome" array are the
# values that will be used to sort the IRC data.
#
SortTitle, SortETitle = IRCArrays[0]
#
# Find which one of the two fchk files is the one with the IRC in the forward
# direction and set it as reference. The data for the reverse direction will be
# added to it.
#
MaxA = max(FChkA[SortTitle]['Array'][1::2])
MaxB = max(FChkB[SortTitle]['Array'][1::2])
if MaxA > MaxB:
    FChkRef = FChkA
    TitlesRef = TitlesA
    FChkSec = FChkB
else:
    FChkRef = FChkB
    TitlesRef = TitlesB
    FChkSec = FChkA
#
# Merge arrays in two steps:
#
# 1. Create a list with all the IRC data recovered from the two fchk files.
#    Each element of the list contains all the data for a given IRC point.
#
IRCData = []
for (ATitle, ETitle) in IRCArrays:
    NEleP = FChkRef[ETitle]['Value']
    # Merge arrays from the two fchk files.
    ArraySum = FChkRef[ATitle]['Array'] + FChkSec[ATitle]['Array']
    # Organize data by IRC point.
    for PStart in range(0, len(ArraySum), NEleP):
        Point = PStart / NEleP
        PData = ArraySum[PStart : PStart + NEleP]
        try:
            IRCData[Point].update({ATitle : PData})
        except IndexError:
            IRCData.append({ATitle : PData})
#
#   Sort points in the merged IRC data list by ascending order of the reaction
#   coordinate value.
#
IRCData.sort(key=lambda IPoint: IPoint[SortTitle][1])
#
# 2. Clear original arrays with IRC data in reference and fill them with the
#    merged and sorted data from above.
#
for (ATitle, ETitle) in IRCArrays:
    FChkRef[ATitle]['Array'] = []
    PastTS = False
    for Point in range(len(IRCData)):
        # Eliminate duplicate entry for TS.
        if IRCData[Point][SortTitle][1] == 0:
            if PastTS:
                continue
            else:
                PastTS = True
        FChkRef[ATitle]['Array'].extend(IRCData[Point][ATitle])
#
# Modify the array with the number of geometries in the IRC to reflect the new
# number after merging and removing the duplicate entry for the TS.
#
NPoints = len(FChkRef[SortTitle]['Array']) / FChkRef[SortETitle]['Value']
FChkRef['IRC Number of geometries']['Array'] = [ NPoints ]
#
# Write formatted checkpoint file with merged IRC data.
#
OutFChk = open(sys.argv[3], 'w')
OutFChk.write(prt_fchk(FChkRef, TitlesRef))
OutFChk.close()
