classdef TDDFT_Single_Excitation_Class_V1_2014a<handle & dynamicprops    %Inheriting from handle class allows the methods to modify object properties(lol that this isnt default)
    
    
    
    %Contains all info on a single excitation from a TDDFT calc
    %   Input is a cell array containing n lines from .log file (Starts at
    %   "Excited State n" and ends at the blank line in the .log file). The
    %   required input is generated by the TDDFT_Output_Class_V1
    
    %excited_states input var = excitation_array entry (i.e 1 row) from TDDFT_Output_Class_V1
    %multiplicity input = integer, obviously the multiplicity of the ground
    %state (ASSUMED OPEN SHELL ARE UNRESTRICTED CALCS). This is required
    %for the normalisation of the contributions to each excitation
    %(normalised to 0.5 for closed shell and 1.0 for open shell)
    
    %*************************HISTORY**************************%
    %   Oct 2016- V1 completed.
    %   Oct 2016 - Addedd the Get_UMO_OMO method.
    %   Apr 2017 - _2014a method. Changed "strip" to "strtrim" for
    %              backwards compat. 
    %              Also made it inherit from handle class
    %   May 2017- Fixed a bug where it didnt split the excitation data
    %             properly when OMO/UMO>99 (since there was no space
    %             between numbere and ->
    
    properties
        
        
        
        %----->Medium raw data(basic analysis done)<-----
        UMO_contribs;   %Struct with fields "orbital" and "fract" (fractional contrib to excitation)
        OMO_contribs;   %Same as UMO_contribs but the GS MOs which contribute
        
        
        
        
        %---->RAW DATA(straight from output file<-------
        energy_ev;  %Energy of excitation in eV
        energy_nm;  %Energy of the excitation in nanometers
        S2_val;   %<S**2>  in output file. Its the s(s+1) of the excited state( 0 is singlet,1 is doublet,2 is triplet etc.)
        osc_strength;   %Oscialltor strength of transition, f value in g09 output
        fs_symm;    %The symmetry of the excited state
        %excitations_data is a struct, fields = OMO,UMO,amplitude. 
        %first two are 1xn cell arrays, third one is 1xn matrix. 
        excitations_data;   
        %%Same format as excitations_data but obviously only for de-excitations (i.e OMO<--UMO)
        %NOTE: for 29B <- 30B OMO=29, UMO = 30B.
        de_excitations_data;    
    end
    
    methods
        function obj = TDDFT_Single_Excitation_Class_V1_2014a(excited_states,multiplicity)
            if nargin>0 %Must do nothing if no args given, otherwise cant initialise object array
                
                amp_sum_error_threshold = 1.03;  %Maximum the normalised sum of amplitudes can add to. Not set to 1 as printing thresholds combined with de-excitations can make our sum >1
                
                %*****SECTION 1****
                %Exctracting info on 1st line of input ("Excited State....")
                first_line_str = strsplit(strtrim(excited_states{1}));
                obj.energy_ev = str2num(first_line_str{5});
                obj.energy_nm = str2num(first_line_str{7});
                obj.osc_strength = str2num(strrep(first_line_str{9},'f=',''));
                obj.S2_val = str2num(strrep(first_line_str{10},'<S**2>=',''));
                obj.fs_symm = first_line_str{4};
                
                %****SECTION 2****
                %Extracting info on the contributions to the transition
                ex_count = 1;
                de_ex_count = 1;
                de_ex_OMO={};
                de_ex_UMO={};
                de_ex_amp={};
                for i = 2:size(excited_states,1)
                    if ~isempty(strfind(excited_states{i},'->'))
                        formd_str = strrep(strrep(excited_states{i},'->','-> '),'->','-> ');
                        formd_str = strrep(excited_states{i},'->',' -> ');

                        t_array = strsplit(strtrim(formd_str));
                        ex_OMO{1,ex_count} = t_array{1};
                        ex_UMO{1,ex_count} = t_array{3};
                        ex_amp{1,ex_count} = [str2num(t_array{4})];
                        ex_count = ex_count+1;
                    end
                    
                    if ~isempty(strfind(excited_states{i},'<-'))
                        formd_str = strrep(excited_states{i},'<-',' <- ');
                        t_array = strsplit(strtrim(formd_str));
                        %t_array = strsplit(strtrim(excited_states{i}));
                        de_ex_OMO{1,de_ex_count} = t_array{1};
                        de_ex_UMO{1,de_ex_count} = t_array{3};
                        de_ex_amp{1,de_ex_count} = [str2num(t_array{4})];
                        de_ex_count = de_ex_count + 1;
                    end
                end
                
                obj.excitations_data = struct('OMO',ex_OMO,...
                    'UMO',ex_UMO,...
                    'amplitude',ex_amp);
                obj.de_excitations_data = struct('OMO',de_ex_OMO,...
                    'UMO',de_ex_UMO,...
                    'amplitude',de_ex_amp);
                
                %****SECTION 3*****
                %Getting fractional contribs of each MO to the excitation
                %Start by getting list of all MOs which contribute
                OMO_list{1} = obj.excitations_data(1).OMO; %All OMOs contributing to excitations data
                UMO_list{1} = obj.excitations_data(1).UMO; %All UMOs contributing to excitations data
                OMO_count=2;
                UMO_count=2;
                
                %All MOs which contribute to the excitations
                for i =2:size(obj.excitations_data,2)
                    if isempty(strfind(strjoin(OMO_list),obj.excitations_data(i).OMO))
                        OMO_list{OMO_count} = obj.excitations_data(i).OMO;
                        OMO_count = OMO_count + 1;
                    end
                    if isempty(strfind(strjoin(UMO_list),obj.excitations_data(i).UMO))
                        UMO_list{UMO_count} = obj.excitations_data(i).UMO;
                        UMO_count = UMO_count + 1;
                    end
                end
                
                %All MOs contributing to the de-exccitations
                for i =2:size(obj.de_excitations_data,2)
                    if isempty(strfind(strjoin(OMO_list),obj.de_excitations_data(i).OMO))
                        OMO_list{OMO_count} = obj.de_excitations_data(i).OMO;
                        OMO_count = OMO_count + 1;
                    end
                    if isempty(strfind(strjoin(UMO_list),obj.de_excitations_data(i).UMO))
                        UMO_list{UMO_count} = obj.de_excitations_data(i).UMO;
                        UMO_count = UMO_count + 1;
                    end
                end

                %Now summing all sqrds amplitudes (de_excitations are
                %subtracted)
                OMO_ampsum = cell(size(OMO_list,1),size(OMO_list,2));
                UMO_ampsum = cell(size(UMO_list,1),size(UMO_list,2));
                
                %Sum of Amplitudes for the OMOs
                for i=1:size(OMO_list,2)    %size,2?
                    ampsqr_sum=0 ;  %sum for current MO (sqr of amplitudes)
                    for j = 1:size(obj.excitations_data,2)  %excitations first (add contribs from here)
                       if strcmpi(obj.excitations_data(j).OMO,OMO_list{i});
                            ampsqr_sum = ampsqr_sum + (obj.excitations_data(j).amplitude^2);
                       end
                    end
                    for j = 1:size(obj.de_excitations_data,2)  %excitations first (add contribs from here)
                        if strcmpi(obj.de_excitations_data(j).OMO,OMO_list{i});
                            ampsqr_sum = ampsqr_sum - (obj.de_excitations_data(j).amplitude^2);
                        end
                    end
                    OMO_ampsum{1,i} = [ampsqr_sum];
                    %Closed/Open shell calcs normalised differently
                    if multiplicity ==1
                        ampsqr_sum = ampsqr_sum*2;
                    end
                end
                
                
                %Sum of Amplitudes for the UMOs
                for i=1:size(UMO_list,2)    %size,2?
                    ampsqr_sum=0 ;  %sum for current MO (sqr of amplitudes)
                    for j = 1:size(obj.excitations_data,2)  %excitations first (add contribs from here)
                        if strcmpi(obj.excitations_data(j).UMO,UMO_list{i});
                            ampsqr_sum = ampsqr_sum + (obj.excitations_data(j).amplitude^2);
                        end
                    end
                    for j = 1:size(obj.de_excitations_data,2)  %excitations first (add contribs from here)
                        if strcmpi(obj.de_excitations_data(j).UMO,UMO_list{i});
                            ampsqr_sum = ampsqr_sum - (obj.de_excitations_data(j).amplitude^2);
                        end
                    end
                   %Closed/Open shell calcs normalised differently
                   if multiplicity ==1
                        ampsqr_sum = ampsqr_sum*2;
                    end
                    UMO_ampsum{1,i} = [ampsqr_sum];
                end

                %Creating the OMO and UMO structures:
                obj.UMO_contribs = struct('orbital',UMO_list,...
                                          'fract',UMO_ampsum);
                obj.OMO_contribs = struct('orbital',OMO_list,...
                                          'fract',OMO_ampsum);
                
                %Error check making sure no contrib > total excitation? 
                %Or at least UMO_ampsums = OMO_ampsums?(the totals)
                if sum(cat(1,obj.OMO_contribs(:).fract)) > amp_sum_error_threshold
                    error(['You have a sum of OMO contribs=', num2str(sum(cat(1,obj.OMO_contribs(:).fract))) 'for excitation with E = ',num2str(obj.energy_ev),'. This might be due to multiplicity being wrong or just a problem with prnting threshold from gaussian']);
                end
                
                if sum(cat(1,obj.OMO_contribs(:).fract)) ~= sum(cat(1,obj.OMO_contribs(:).fract))
                    error('UMO/OMO contribs have different amplitude sums, must be something wrong with code');
                end
                
                
            end
        end
        
        %orb_contrib = the fractional amount that OMO->UMO transition
        %               contributed to this excitation
        %OMO = string representation of the OMO of interest (e.g '30B')
        %UMO = string representation of the UMO of interest (e.g '24B')
        function [orb_contrib] = Get_UMO_OMO(obj,OMO,UMO)
            rel_excitation_amp = 0;
            rel_de_excitation_amp = 0;
            %Gets the excitation amplitude (if present)
            for i = 1:size(obj.excitations_data,2)
                if strcmp(obj.excitations_data(1,i).OMO,OMO) ...
                        && strcmp(obj.excitations_data(1,i).UMO,UMO)
                    rel_excitation_amp = obj.excitations_data(1,i).amplitude^2;
                    break;
                end
                
            end
            %Gets the de_excitation amplitude (if present)
            for i = 1:size(obj.de_excitations_data,2)
                if strcmp(obj.de_excitations_data(1,i).OMO,OMO) ...
                        && strcmp(obj.de_excitations_data(1,i).UMO,UMO)
                    rel_de_excitation_amp = obj.de_excitations_data(1,i).amplitude^2;
                    break;
                end
            end
            %Difference between excitation and de-excitation amplitude
            orb_contrib = rel_excitation_amp - rel_de_excitation_amp
        end
    end
end
    


