classdef Sumviz_File_Obj_V2 < handle %Inherit from handle, allows object properties to be modified by methods
    %UNTITLED holds the info in a .sumviz file + provides methods to help
    %summarise the data
    %   Detailed explanation goes here
    
    %HISTORY
    %V1 Finished at unknown time
    %V2 (July 2017) - Made it so code works on either mac or windows (not
    %                 just windows)
    
    
    properties
        base_fname; %Filename without full path or extenion (x/y/file.sumviz--->file)
        cp_matrix;  %Contains AIM_crit_point objects. (nx1 array)
        geometry;  %1st col=element symbols + atomic labels, 2nd-->4th col = x/y/z co-ordinates
        atomic_charges; %1st col=element symbol+atom label, 2nd col=AIM atomic charges
        
        %INTERMEDIATE PROPERTIES(mainly for debuggin)
        rel_lines_cp_matrix;   % nx1 cell array, each entry is cell array containing all lines relevant to a critical point
        
        
    end
    
    methods
        %CONSTRUCTOR:Obviously just extracts all the info from the file
        function obj = Sumviz_File_Obj_V2(filename)
           if nargin>0  %Required to do nothing if not given a arg<--- allows initialisation of array objects
               
               %****SECTION 1****%
               %Put file in 1xn cell array(column), each line=1 row. also put the
               %base filename as one of the object properties
               fobject1=fopen(filename);
               full_file = fileread(filename);
               full_file = strsplit(full_file,'\n')';
               fclose(fobject1);
               [~,tstring] = fileparts(filename);
               obj.base_fname = strrep(tstring,'.log','');
               
               
               %*****SECTION 2****%
               %Extract geometries + ,
               obj.geometry = cell(1,4);
               counter = 1;	%Counts current column in geometry
               copy_geom=0;	%Will only copy over geom when=1(triggers at start of geom section)
               geom_start = 0; %Triggers when hit "Nuclear Charges and Cartesian Coordinates"
               atom_label_regexp = '[A-Z]+[0-9]'; %The reg. expression for an atomic label (e.g H6,S12,Be8)
               for i=1:size(full_file,1)
                   curr_line = strtrim(full_file{i,1});
                   if ~isempty(strfind(curr_line,'Nuclear Charges and Cartesian Coordinates')) %Triggers start of geometry extract section
                       geom_start = 1;
                   end
                   if geom_start==1
                       if ~isempty(regexp(curr_line,atom_label_regexp)) %Only extract geometry when line starts [A-Z]+\d
                           copy_geom = 1
                           geom_line = strsplit(curr_line)
                           obj.geometry{counter,1} = geom_line{1}; %Atom label
                           obj.geometry{counter,2} = geom_line{3}; %x
                           obj.geometry{counter,3} = geom_line{4}; %y
                           obj.geometry{counter,4} = geom_line{5}; %z
                           counter = counter + 1;
                       end
                   end
                   if geom_start==1 && copy_geom==1 && isempty(regexp(curr_line,atom_label_regexp)) %Trigger to stop extracting geometry
                       copy_geom = 0;
                       geom_start = 0;
                       break;
                   end
               end
               
               %*****SECTION 3****
               %Extract Atomic Charges, basically the same as geometry
               %extraction but w/different variables
               obj.atomic_charges = cell(1,2);
               counter = 1;	%Counts current column in geometry
               copy_att_charge=0;	%Will only copy over geom when=1(triggers at start of geom section)
               att_charge_start = 0; %Triggers when hit "Nuclear Charges and Cartesian Coordinates"
               atom_label_regexp = '[A-Z]+[0-9]'; %The reg. expression for an atomic label (e.g H6,S12,Be8)
               for i=1:size(full_file,1)
                   curr_line = strtrim(full_file{i,1});
                   if ~isempty(strfind(curr_line,'Some Atomic Properties')) %Triggers start of geometry extract section
                       att_charge_start = 1;
                   end
                   if att_charge_start==1
                       if ~isempty(regexp(curr_line,atom_label_regexp)) %Only extract geometry when line starts [A-Z]+\d
                           copy_att_charge = 1;
                           charge_line = strsplit(curr_line);
                           obj.atomic_charges{counter,1} = charge_line{1}; %Atom label
                           obj.atomic_charges{counter,2} = str2num(charge_line{2}); %charge
                           counter = counter + 1;
                       end
                   end
                   if att_charge_start==1 && copy_att_charge==1 && isempty(regexp(curr_line,atom_label_regexp)) %Trigger to stop extracting geometry
                       copy_att_charge = 0;
                       att_charge_start = 0;
                       break;
                   end
               end

                              
               
               %****SECTION 4****%
               %Getting all the crit point sections into a cell array
               start_pattern = 'CP#';    %This pattern appears on first line for each critical point
               final_pattern = 'DivStress'; %Pattern appears on last line i care about in the crit points
               obj.rel_lines_cp_matrix = cell(1,1); %Each row holds the relevant lines for the critical point objects
               counter = 1; %Tracks the value of rel_lines_matrix we're on 
               copy_cp = 0; %When = 1 each line of full_file is copied to rel_lines_cp_matrix
               for i=1:size(full_file,1)
                    if ~isempty(strfind(full_file{i,1},start_pattern))
                        copy_cp=1;
                        curr_start_index = i;   %Indice where current crit. point info starts
                    end
                   
                    if ~isempty(strfind(full_file{i,1},final_pattern)) && copy_cp==1 ;
                        copy_cp = 0;
                        obj.rel_lines_cp_matrix{counter,1} = full_file(curr_start_index:i,1);
                        counter = counter + 1;
                    end
                   
               end
               
               
            %****SECTION 5****%
            %Create the cp_matrix, holds all the info on each cp in the AIM
            %sumviz file
            cp_matrix(size(obj.rel_lines_cp_matrix,1),1) = AIM_cp_class;    %For some reason i cant do this for "obj.cp_matrix"
            
            for i =1:size(obj.rel_lines_cp_matrix,1)
               cp_matrix(i,1) = AIM_cp_class(obj.rel_lines_cp_matrix{i,1});
            end
            obj.cp_matrix = cp_matrix;
            
           end
        end
        
        
        %Returns a list of indices (for cp_matrix) which contain
        % critical points involving atoms from BOTH frag1 and frag2
        % frags are defined by lists of the atomic numbers which make up 
        % the fragment. E.g get_rel_crit_points('1-12,15-20','13-14') finds
        % critical points between fragments with atoms '1-12,15-20' and '13-14'
        
        function [rel_indice_list] = get_rel_crit_points(obj,frag1,frag2)
            %Finding all the critical points involving at least 1 atom from
            %both frag1 and frag2
            frag1_atoms = obj.Split_Number_List(frag1)';
            frag2_atoms = obj.Split_Number_List(frag2)';
            search_matrix = cell(size(frag1_atoms,1) * size(frag2_atoms,1),2);  %Stores Every pair of atoms between the two fragments (1 pair per row, 1 atom per column)
            counter = 1; %Tracks position in search_matrix
            for i =1:size(frag1_atoms,1)
                for j=1:size(frag2_atoms,1)
                    search_matrix{counter,1} = obj.geometry{frag1_atoms{i}};  %From frag1
                    search_matrix{counter,2} = obj.geometry{frag2_atoms{j}};  %From frag2
                    counter = counter + 1;
                end
            end
            %Now finding all relevant indices in obj.cp_matrix
            rel_indice_list = NaN(1,1)  %Each row contains an indice for cp_matrix which has a critical point of interest
            counter = 1;
            for i=1:size(obj.cp_matrix,1)
                for j=1:size(search_matrix,1)
                    if ~isempty( find(strcmp(obj.cp_matrix(i,1).atoms,search_matrix{j,1}))) && ~isempty( find(strcmp(obj.cp_matrix(i,1).atoms,search_matrix{j,2})))
                        rel_indice_list(counter,1) = i
                        counter = counter + 1;
                    end
                end
            end
        end
        
        
        
        
        %Returns the sum of rho values for all bcp's between frag1 and
        %frag2 (see get_rel_crit_points for notation for entering
        %frag1/frag2 (There both str containig atom lists)
        %OUTPUT:
        %sum_rho = the sum of rho value for all inter-frag interactions
        %max_rho = the maximum rho value for an interfrag bcp
        %max_rho_info = str containing the atoms which contribute to
        %                the max_rho bcp
        %number_bcp = Total number of bcp's between the fragments
        %bcp_indice_list = array containing indices (in obj.cp_matrix) with the
        %               interfrag bcp's (nx1 array)
        function [sum_rho,max_rho,max_rho_info,number_bcp,bcp_indice_list] = bcp_info_2frags(obj,frag1,frag2)
            %Find all relevant critical points:
            indice_list =  obj.get_rel_crit_points(frag1,frag2); %Contains ALL cp involving at least 1 atom from each fragment
            bcp_indice_list = NaN(1,1);  %Contains only bcp interfrag cp's (rather than all interfrag cp's)
            counter = 1;    %Tracks how many indices we've added to bcp_indice_list
            if isnan(indice_list(1,1))  %If no BCPs found then we just assign 0 to lots of things
                sum_rho = 0;
                max_rho = 0;
                max_rho_info = '';
                number_bcp = 0;
                bcp_indice_list=NaN;
            end
            
            %Extracting only BCP's assuming theres some type of CP
            if ~isnan(indice_list(1,1))
            for i=1:size(indice_list)
                if strcmp('BCP',obj.cp_matrix(indice_list(i,1)).type)
                    bcp_indice_list(counter,1) = indice_list(i,1);
                    counter = counter + 1;
                end
            end
            end
            
           if isnan(bcp_indice_list(1,1))  %If no BCPs found then we just assign 0 to lots of things
                sum_rho = 0;
                max_rho = 0;
                max_rho_info = '';
                number_bcp = 0;
                bcp_indice_list=NaN;
            end

            %--->Find all the relevant properties from the cp indices
            %(Assuming we have BCPs between fragments)
            if ~isnan(bcp_indice_list(1,1))
            sum_rho = 0;
            max_rho_indice = bcp_indice_list(1,1) %Initialise to first bcp, then update if stronger bcp found
            max_rho = obj.cp_matrix(bcp_indice_list(1,1),1).rho
            for i=1:size(bcp_indice_list,1)
                sum_rho = sum_rho +  obj.cp_matrix(bcp_indice_list(i,1),1).rho
                if obj.cp_matrix(bcp_indice_list(i,1),1).rho > max_rho
                    max_rho_indice = bcp_indice_list(i,1)
                    max_rho = obj.cp_matrix(max_rho_indice,1).rho
                end
                
            end
            max_rho_info = strjoin(obj.cp_matrix(max_rho_indice,1).atoms,'-')
            number_bcp = size(bcp_indice_list,1);
            %
            %max_rho = max([obj.cp_matrix(:,1).rho]) %This would find the max rho for the entire object array
            end
            
            
            
            
            
            
        end
        
        
        % Returns the sum of AIM atomic charges for the list of atoms in
        % "frag". "frag" input is a string containing a lst of atoms in
        % format '1-4,6,9-12' (for example)
        function [frag_charge] = get_fragment_charge(obj,frag)
            frag_atom_list = obj.Split_Number_List(frag)'; %nx1 cell array, each entry = 1 number
            frag_charge = 0;
            for i=1:size(frag_atom_list)
                frag_charge = frag_charge + obj.atomic_charges{frag_atom_list{i,1},2};
            end
        end
        
        
        
    end
    
    methods (Static)
        %Takes the str list of numbers (e.g '3-5,6,9,1-19') and splits it into
        %cell array (split list, 1xn) so that each number is in its own cell
        function [split_list] = Split_Number_List(number_list)
            split_numb_list = strsplit(number_list,','); %each comma delim value in own cell
            
            %First loop works out the array size needed, so it only needs initialisation once (assuming array resizing takes aggggggeeeeees)
            total_array_size=0;
            for i=1:size(split_numb_list,2);
                split_numb_list{i}=strtrim(split_numb_list{i});	%remove trailing/leading whitepsace
                if ~isempty(strfind(split_numb_list{i},'-'));
                    dash_indice= strfind(split_numb_list{i},'-'); %char position of the dash
                    tstring = split_numb_list{i};
                    start_pos = str2num(tstring(1:dash_indice-1));
                    end_pos = str2num(tstring(dash_indice+1:end));
                    total_array_size=total_array_size + (end_pos-start_pos) +1;%+1 as we need start AND end (i.e its inclusive)
                else
                    total_array_size = total_array_size + 1;
                end
            end
            
            %Initialising+populating array with numbers of all atoms to use
            split_list=zeros(1,total_array_size);
            counter=1;	%tracks number of split_list we're on
            for i=1:size(split_numb_list,2);
                split_numb_list{i}=strtrim(split_numb_list{i});	%remove trailing/leading whitepsace
                tstring = split_numb_list{i};	%string for current cell
                if ~isempty(strfind(split_numb_list{i},'-'));
                    dash_indice= strfind(split_numb_list{i},'-'); %char position of the dash
                    start_pos = str2num(tstring(1:dash_indice-1));	%first number to put in split_list
                    end_pos = str2num(tstring(dash_indice+1:end));	%Go up to this number in split_list
                    split_list(1,counter:counter+((end_pos-start_pos))) = start_pos:end_pos;
                    counter=counter+((end_pos-start_pos))+1;
                else
                    split_list(1,counter) = str2num(tstring);
                    counter=counter+1;
                end
            end
            counter=1;
            split_list=num2cell(split_list);    %Want a cell array for this script.....
            
        end
        
        % object_array = Sumviz_File_Obj_V2.Create_sumviz_obj_array_folder to
        % run
        %Takes pull path to a folder (folder input var ldo) and returns an object array
        %containing Sumviz_File_Obj_V2 for all .sumviz files in folder
        %Also runs an error check to make sure atomic numbering is the same
        %for all the files, will throw error otherwise
        function [AIM_obj_array] = Create_sumviz_obj_array_folder(folder)
           %Creating the object array
           file_list = dir(fullfile(folder,'*.sumviz')); %nx1 structure containing all relevant file names
           AIM_obj_array(size(file_list,1),1) = Sumviz_File_Obj_V2;
           for file = 1:size(file_list,1)
               AIM_obj_array(file,1) = Sumviz_File_Obj_V2(fullfile(folder,file_list(file,1).name));
           end
           %Running error check on geometry
           %First check number of atoms is constant throughout:
           numb_atoms = NaN(size(AIM_obj_array,1),1)
           for i=1:size(numb_atoms,1)
               numb_atoms(i,1) = size(AIM_obj_array(i,1).geometry,1)
           end
           %numb_atoms = numb_atoms - numb_atoms(1,1)
           if any(numb_atoms- numb_atoms(1,1))
               for i=1:size(AIM_obj_array,1)
                   disp([num2str(size(AIM_obj_array(i,1).geometry,1)) ' Atoms in '  AIM_obj_array(i,1).base_fname])
               end
               error(['Inconsistent Number of atoms in folder ' folder])
           end
           %Now check geometry is constant throughout
           all_geoms = cell(numb_atoms(1,1),size(AIM_obj_array,1));	%Each atom=1 row, each file = 1 col
           for j=1:size(all_geoms,2)
               all_geoms(:,j) = AIM_obj_array(j,1).geometry(:,1)
           end
           
           for j=1:size(all_geoms,2)
               if ~all(strcmp(all_geoms(:,1),all_geoms(:,j)))
                   error(['Atomic Numbering is not constant within folder ' folder '.Inconsistencies in files  ' AIM_obj_array(1,1).base_fname  ' and ' AIM_obj_array(j,1).base_fname])
               end
           end
        end
        
    end
    
end

